import 'package:flutter/material.dart';
import 'package:get/get.dart';
import 'package:viserpay/core/helper/date_converter.dart';
import 'package:viserpay/core/helper/string_format_helper.dart';
import 'package:viserpay/core/utils/dimensions.dart';
import 'package:viserpay/core/utils/my_color.dart';
import 'package:viserpay/core/utils/my_strings.dart';
import 'package:viserpay/core/utils/style.dart';
import 'package:viserpay/data/controller/reward/reward_controller.dart';
import 'package:viserpay/view/components/app-bar/custom_appbar.dart';
import 'package:viserpay/view/components/divider/custom_divider.dart';
import 'package:viserpay/view/screens/reward/widget/reward_card.dart';
import 'package:viserpay/view/screens/reward/widget/reward_earned_widget.dart';
import 'package:viserpay/view/screens/reward/widget/reward_used_widget.dart';

class UserRewardHistory extends StatefulWidget {
  const UserRewardHistory({super.key});

  @override
  State<UserRewardHistory> createState() => _UserRewardHistoryState();
}

class _UserRewardHistoryState extends State<UserRewardHistory> with SingleTickerProviderStateMixin {
  late TabController _tabController;
  @override
  void initState() {
    super.initState();
    _tabController = TabController(length: 3, vsync: this);
  }

  List<String> title = [
    MyStrings.earned,
    MyStrings.used,
    MyStrings.expired,
  ];

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      backgroundColor: MyColor.screenBgColor,
      appBar: CustomAppBar(title: MyStrings.reward, isTitleCenter: true),
      body: GetBuilder<RewardController>(builder: (controller) {
        return Padding(
          padding: Dimensions.screenPaddingHV,
          child: Column(
            children: [
              Container(
                width: double.infinity,
                padding: EdgeInsets.symmetric(horizontal: 20, vertical: 30),
                decoration: BoxDecoration(color: MyColor.colorWhite, borderRadius: BorderRadius.circular(Dimensions.mediumRadius)),
                child: Column(
                  children: [
                    Text(MyStrings.availablePoints.tr, style: regularDefault.copyWith()),
                    Text(StringConverter.formatNumber(controller.user?.totalPoints ?? "0.0", precision: 0), style: semiBoldDefault.copyWith(fontSize: 35)),
                    Text("${MyStrings.lastUpdatedAt.tr} ${DateConverter.convertIsoToString(controller.user?.updatedAt ?? "")}", style: regularDefault.copyWith()),
                  ],
                ),
              ),
              SizedBox(height: 10),
              Container(
                decoration: BoxDecoration(color: MyColor.colorWhite, borderRadius: BorderRadius.circular(Dimensions.defaultRadius)),
                child: TabBar(
                  controller: _tabController,
                  labelColor: MyColor.primaryColor,
                  dividerColor: MyColor.getContentTextColor().withValues(alpha: 0.1),
                  unselectedLabelColor: MyColor.getContentTextColor(),
                  indicatorColor: MyColor.primaryColor,
                  isScrollable: true,
                  tabAlignment: TabAlignment.center,
                  labelPadding: EdgeInsetsDirectional.symmetric(vertical: Dimensions.space5, horizontal: Dimensions.space10),
                  tabs: List.generate(title.length, (index) => Tab(text: title[index].tr)),
                  onTap: (value) {},
                ),
              ),
              Expanded(
                flex: 1,
                child: Container(
                  color: MyColor.colorWhite,
                  child: TabBarView(
                    physics: AlwaysScrollableScrollPhysics(parent: BouncingScrollPhysics()),
                    controller: _tabController,
                    viewportFraction: 1,
                    children: [
                      EarnedList(),
                      UsedPointList(),
                      ExpiredList(),
                    ],
                  ),
                ),
              ),
            ],
          ),
        );
      }),
    );
  }
}

class EarnedList extends StatelessWidget {
  const EarnedList({super.key});

  @override
  Widget build(BuildContext context) {
    return GetBuilder<RewardController>(
      builder: (controller) {
        return ListView.builder(
          itemCount: controller.earnedPoint.length,
          itemBuilder: (context, index) {
            return Container(
              padding: EdgeInsets.symmetric(vertical: Dimensions.space5),
              decoration: BoxDecoration(color: MyColor.colorWhite, borderRadius: BorderRadius.circular(Dimensions.defaultRadius)),
              child: RewardEarnedCard(
                reward: controller.earnedPoint[index],
                currencySym: controller.rewardRepo.apiClient.getCurrencyOrUsername(isCurrency: true, isSymbol: true),
              ),
            );
          },
        );
      },
    );
  }
}

class UsedPointList extends StatelessWidget {
  const UsedPointList({super.key});

  @override
  Widget build(BuildContext context) {
    return GetBuilder<RewardController>(builder: (controller) {
      return ListView.builder(
        itemCount: controller.userPoint.length,
        itemBuilder: (context, index) {
          final reward = controller.userPoint[index];
          return RewardUsedCard(
            reward: reward,
            currencySym: controller.rewardRepo.apiClient.getCurrencyOrUsername(isCurrency: true, isSymbol: true),
          );
        },
      );
    });
  }
}

class ExpiredList extends StatelessWidget {
  const ExpiredList({super.key});

  @override
  Widget build(BuildContext context) {
    return GetBuilder<RewardController>(builder: (controller) {
      return ListView.builder(
        itemCount: controller.expirePoint.length,
        itemBuilder: (context, index) {
          final reward = controller.expirePoint[index];
          return Container(
            padding: EdgeInsets.symmetric(vertical: Dimensions.space5, horizontal: Dimensions.space10),
            decoration: BoxDecoration(color: MyColor.colorWhite, borderRadius: BorderRadius.circular(Dimensions.defaultRadius)),
            child: Column(
              children: [
                Row(
                  children: [
                    Expanded(
                      flex: 2,
                      child: Column(
                        crossAxisAlignment: CrossAxisAlignment.start,
                        children: [
                          Text(MyStrings.expired.tr, style: regularDefault.copyWith()),
                          Text(reward.month ?? "", style: lightDefault.copyWith()),
                        ],
                      ),
                    ),
                    Text("-${reward.totalPoints}", style: semiBoldDefault.copyWith(color: MyColor.redCancelTextColor)),
                  ],
                ),
                CustomDivider(color: MyColor.colorGrey1, space: 5, onlyBottom: true),
              ],
            ),
          );
        },
      );
    });
  }
}
